<?php
require_once 'config/database.php';
$page_title = "Email Reports";
require_once 'includes/header.php';

// Get statistics
$totalCampaigns = $conn->query("SELECT COUNT(*) as total FROM email_campaigns")->fetch_assoc()['total'];
$totalSent = $conn->query("SELECT COUNT(*) as total FROM email_logs WHERE status='sent'")->fetch_assoc()['total'];
$totalFailed = $conn->query("SELECT COUNT(*) as total FROM email_logs WHERE status='failed'")->fetch_assoc()['total'];
$successRate = $totalSent > 0 ? round(($totalSent / ($totalSent + $totalFailed)) * 100, 2) : 0;

// Get campaign reports
$campaigns = $conn->query("
    SELECT c.*, t.name as template_name,
           (SELECT COUNT(*) FROM email_logs WHERE campaign_id = c.id AND status='sent') as sent_count,
           (SELECT COUNT(*) FROM email_logs WHERE campaign_id = c.id AND status='failed') as failed_count
    FROM email_campaigns c
    LEFT JOIN email_templates t ON c.template_id = t.id
    ORDER BY c.created_at DESC
");

// Get recent logs
$recentLogs = $conn->query("
    SELECT l.*, r.email as receiver_email
    FROM email_logs l
    LEFT JOIN email_receivers r ON l.receiver_email = r.email
    ORDER BY l.sent_at DESC
    LIMIT 50
");
?>

<div class="container-fluid">
    <h1 class="mb-4">Email Reports & Analytics</h1>
    
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-3 col-sm-6 mb-3">
            <div class="stat-card text-center" style="border-top: 5px solid #667eea;">
                <i class="fas fa-bullhorn text-primary"></i>
                <h3><?php echo number_format($totalCampaigns); ?></h3>
                <p class="text-muted">Total Campaigns</p>
            </div>
        </div>
        <div class="col-md-3 col-sm-6 mb-3">
            <div class="stat-card text-center" style="border-top: 5px solid #10b981;">
                <i class="fas fa-check-circle text-success"></i>
                <h3><?php echo number_format($totalSent); ?></h3>
                <p class="text-muted">Sent Emails</p>
            </div>
        </div>
        <div class="col-md-3 col-sm-6 mb-3">
            <div class="stat-card text-center" style="border-top: 5px solid #f59e0b;">
                <i class="fas fa-times-circle text-warning"></i>
                <h3><?php echo number_format($totalFailed); ?></h3>
                <p class="text-muted">Failed Emails</p>
            </div>
        </div>
        <div class="col-md-3 col-sm-6 mb-3">
            <div class="stat-card text-center" style="border-top: 5px solid #8b5cf6;">
                <i class="fas fa-chart-line text-purple"></i>
                <h3><?php echo $successRate; ?>%</h3>
                <p class="text-muted">Success Rate</p>
            </div>
        </div>
    </div>
    
    <div class="row">
        <!-- Campaign Reports -->
        <div class="col-lg-8">
            <div class="card shadow-sm mb-4">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="fas fa-list-alt me-2"></i>Campaign Reports</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Subject</th>
                                    <th>Template</th>
                                    <th>Sent/Failed</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while($campaign = $campaigns->fetch_assoc()): 
                                    $sent = $campaign['sent_count'] ?? 0;
                                    $failed = $campaign['failed_count'] ?? 0;
                                    $total = $sent + $failed;
                                    $percentage = $total > 0 ? round(($sent / $total) * 100) : 0;
                                ?>
                                <tr>
                                    <td>#<?php echo $campaign['id']; ?></td>
                                    <td><?php echo htmlspecialchars($campaign['subject']); ?></td>
                                    <td><?php echo htmlspecialchars($campaign['template_name'] ?? 'N/A'); ?></td>
                                    <td>
                                        <div class="progress" style="height: 20px;">
                                            <div class="progress-bar bg-success" style="width: <?php echo $percentage; ?>%">
                                                <?php echo $sent; ?>
                                            </div>
                                            <?php if ($failed > 0): ?>
                                            <div class="progress-bar bg-danger" style="width: <?php echo 100 - $percentage; ?>%">
                                                <?php echo $failed; ?>
                                            </div>
                                            <?php endif; ?>
                                        </div>
                                        <small class="text-muted"><?php echo $sent; ?> sent, <?php echo $failed; ?> failed</small>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?php 
                                            echo $campaign['status'] == 'completed' ? 'success' : 
                                                   ($campaign['status'] == 'sending' ? 'warning' : 
                                                   ($campaign['status'] == 'failed' ? 'danger' : 'secondary'));
                                        ?>">
                                            <?php echo ucfirst($campaign['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('Y-m-d H:i', strtotime($campaign['created_at'])); ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-outline-info" onclick="viewCampaignDetails(<?php echo $campaign['id']; ?>)">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Recent Activity -->
        <div class="col-lg-4">
            <div class="card shadow-sm">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="fas fa-history me-2"></i>Recent Activity</h5>
                </div>
                <div class="card-body">
                    <div style="max-height: 400px; overflow-y: auto;">
                        <?php while($log = $recentLogs->fetch_assoc()): ?>
                        <div class="mb-3 pb-3 border-bottom">
                            <div class="d-flex justify-content-between">
                                <small class="text-muted">
                                    <?php echo date('H:i', strtotime($log['sent_at'])); ?>
                                </small>
                                <span class="badge bg-<?php echo $log['status'] == 'sent' ? 'success' : 'danger'; ?>">
                                    <?php echo ucfirst($log['status']); ?>
                                </span>
                            </div>
                            <div class="text-truncate">
                                <strong><?php echo htmlspecialchars($log['subject']); ?></strong>
                            </div>
                            <div class="text-truncate small">
                                To: <?php echo htmlspecialchars($log['receiver_email']); ?>
                            </div>
                            <?php if ($log['status'] == 'failed' && $log['error_message']): ?>
                            <div class="small text-danger mt-1">
                                <i class="fas fa-exclamation-triangle"></i>
                                <?php echo substr(htmlspecialchars($log['error_message']), 0, 50); ?>...
                            </div>
                            <?php endif; ?>
                        </div>
                        <?php endwhile; ?>
                    </div>
                </div>
            </div>
            
            <!-- Export Options -->
            <div class="card shadow-sm mt-4">
                <div class="card-header bg-white">
                    <h5 class="mb-0"><i class="fas fa-download me-2"></i>Export Reports</h5>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <a href="export.php?type=campaigns" class="btn btn-outline-primary">
                            <i class="fas fa-file-csv me-2"></i> Export Campaigns (CSV)
                        </a>
                        <a href="export.php?type=logs" class="btn btn-outline-success">
                            <i class="fas fa-file-excel me-2"></i> Export Email Logs (CSV)
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Campaign Details Modal -->
<div class="modal fade" id="campaignModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Campaign Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="campaignDetails">
                Loading...
            </div>
        </div>
    </div>
</div>

<script>
function viewCampaignDetails(campaignId) {
    fetch('ajax/get_campaign_details.php?id=' + campaignId)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                let html = `
                    <h6>Campaign #${data.campaign.id}</h6>
                    <table class="table table-bordered">
                        <tr>
                            <th width="30%">Subject</th>
                            <td>${data.campaign.subject}</td>
                        </tr>
                        <tr>
                            <th>Template</th>
                            <td>${data.campaign.template_name}</td>
                        </tr>
                        <tr>
                            <th>Status</th>
                            <td><span class="badge bg-${data.campaign.status == 'completed' ? 'success' : data.campaign.status == 'sending' ? 'warning' : 'danger'}">${data.campaign.status}</span></td>
                        </tr>
                        <tr>
                            <th>Sent Date</th>
                            <td>${data.campaign.sent_at || 'Not sent yet'}</td>
                        </tr>
                        <tr>
                            <th>Total Receivers</th>
                            <td>${data.campaign.receivers_count}</td>
                        </tr>
                        <tr>
                            <th>Success Rate</th>
                            <td>
                                <div class="progress">
                                    <div class="progress-bar bg-success" style="width: ${data.campaign.sent_count / data.campaign.receivers_count * 100}%">
                                        ${data.campaign.sent_count} sent
                                    </div>
                                    <div class="progress-bar bg-danger" style="width: ${data.campaign.failed_count / data.campaign.receivers_count * 100}%">
                                        ${data.campaign.failed_count} failed
                                    </div>
                                </div>
                            </td>
                        </tr>
                    </table>
                    
                    <h6 class="mt-4">Recent Logs</h6>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Email</th>
                                    <th>Status</th>
                                    <th>Time</th>
                                    <th>Error</th>
                                </tr>
                            </thead>
                            <tbody>`;
                
                data.logs.forEach(log => {
                    html += `
                        <tr>
                            <td>${log.receiver_email}</td>
                            <td><span class="badge bg-${log.status == 'sent' ? 'success' : 'danger'}">${log.status}</span></td>
                            <td>${log.sent_at}</td>
                            <td>${log.error_message || '-'}</td>
                        </tr>`;
                });
                
                html += `</tbody></table></div>`;
                document.getElementById('campaignDetails').innerHTML = html;
                
                const modal = new bootstrap.Modal(document.getElementById('campaignModal'));
                modal.show();
            }
        });
}
</script>

<?php require_once 'includes/footer.php'; ?>